<?php

namespace App\Http\Controllers\api;

use Illuminate\Http\Request;
use App\Models\RiskEvaluation;
use App\Models\Risk;
use App\Http\Controllers\Controller;

class RiskEvaluationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
        $riskEvaluations = RiskEvaluation::all();
        return response()->json($riskEvaluations);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
        // Validar los datos recibidos en la solicitud
        $request->validate([
            'risks_id' => 'required|exists:risks,id',
            'type' => 'required|in:inherent,residual',
            'index_a' => 'required|integer',
            'index_b' => 'required|integer',
            'index_c' => 'required|integer',
            'index_d' => 'required|integer',
            // Los campos 'probability', 'ps' y 'NV' no necesitan validación ya que se calcularán automáticamente
            'signification' => 'required|integer',
            'actions' => 'nullable|string|max:200',
        ]);

        // Calcular el valor de 'probability' a partir de 'index_a', 'index_b', 'index_c' y 'index_d'
        $probability = $request->index_a + $request->index_b + $request->index_c + $request->index_d;

        // Calcular el valor de 'ps' a partir de 'probability' y 'severity'
        $ps = $probability * $request->severity;

        switch ($ps) {
            case 4:
                $NV = "TR";
                break;
            case $ps >= 5 && $ps <= 8:
                $NV = "TO";
                break;
            case $ps >= 9 && $ps <= 16:
                $NV = "M";
                break;
            case $ps >= 17 && $ps <= 24:
                $NV = "IM";
                break;
            case $ps >= 25 && $ps <= 36:
                $NV = "IT";
                break;
        }

        // Crear un nuevo registro de risk_evaluation con los datos proporcionados y los calculados
        $riskEvaluation = RiskEvaluation::create([
            'risks_id' => $request->risks_id,
            'type' => $request->type,
            'index_a' => $request->index_a,
            'index_b' => $request->index_b,
            'index_c' => $request->index_c,
            'index_d' => $request->index_d,
            'probability' => $probability,
            'severity' => $request->severity,
            'ps' => $ps,
            'NV' => $NV,
            'signification' => $request->signification,
            'actions' => $request->actions,
        ]);


        // Devolver una respuesta JSON indicando que el registro fue creado exitosamente
        return response()->json(['message' => 'Risk evaluation created successfully', 'data' => $riskEvaluation], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        //
        $riskEvaluation = RiskEvaluation::find($id);
        if (!$riskEvaluation) {
            return response()->json(['message' => 'RiskEvaluation not found'], 404);
        }
        return response()->json($riskEvaluation);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        //
        // Buscar el registro a actualizar
        $riskEvaluation = RiskEvaluation::find($id);

        // Verificar si el registro existe
        if (!$riskEvaluation) {
            return response()->json(['message' => 'Risk Evaluation not found'], 404);
        }

        // Validar los datos recibidos en la solicitud
        $request->validate([
            'risks_id' => 'required|exists:risks,id',
            'type' => 'required|in:inherent,residual',
            'index_a' => 'required|integer',
            'index_b' => 'required|integer',
            'index_c' => 'required|integer',
            'index_d' => 'required|integer',
            'signification' => 'required|integer',
            'actions' => 'nullable|string|max:200',
        ]);

        // Calcular el valor de 'probability' a partir de 'index_a', 'index_b', 'index_c' y 'index_d'
        $probability = $request->index_a + $request->index_b + $request->index_c + $request->index_d;

        // Calcular el valor de 'ps' a partir de 'probability' y 'severity'
        $ps = $probability * $riskEvaluation->severity;

        // Calcular el valor de 'NV' basado en 'ps'
        switch (true) {
            case $ps == 4:
                $NV = "TR";
                break;
            case $ps >= 5 && $ps <= 8:
                $NV = "TO";
                break;
            case $ps >= 9 && $ps <= 16:
                $NV = "M";
                break;
            case $ps >= 17 && $ps <= 24:
                $NV = "IM";
                break;
            case $ps >= 25 && $ps <= 36:
                $NV = "IT";
                break;
            default:
                $NV = null; // Valor por defecto si no se cumple ninguna condición
        }

        // Actualizar los datos del registro
        $riskEvaluation->update([
            'risks_id' => $request->risks_id,
            'type' => $request->type,
            'index_a' => $request->index_a,
            'index_b' => $request->index_b,
            'index_c' => $request->index_c,
            'index_d' => $request->index_d,
            'probability' => $probability,
            'ps' => $ps,
            'NV' => $NV,
            'signification' => $request->signification,
            'actions' => $request->actions,
        ]);

        // Retornar una respuesta indicando que el registro fue actualizado exitosamente
        return response()->json(['message' => 'Risk Evaluation updated successfully', 'risk_evaluation' => $riskEvaluation]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        //
        // Buscar el registro a eliminar
        $riskEvaluation = RiskEvaluation::find($id);

        // Verificar si el registro existe
        if (!$riskEvaluation) {
            return response()->json(['message' => 'Risk Evaluation not found'], 404);
        }

        // Eliminar el registro
        $riskEvaluation->delete();

        // Retornar una respuesta indicando que el registro fue eliminado exitosamente
        return response()->json(['message' => 'Risk Evaluation deleted successfully']);
    }

    public function riskHeatmap()
    {
        // Consultar los datos de evaluación de riesgos
        $risks = RiskEvaluation::select('probability', 'severity', 'actions')->get();

        // Transformar los datos en un formato adecuado para el mapa de calor
        $data = $risks->map(function ($risk) {
            return [
                'probability' => $risk->probability,
                'severity' => $risk->severity,
                'actions' => $risk->actions,
            ];
        });

        // Devolver los datos en formato JSON
        return response()->json($data);
    }

    public function getAllDataByRisk()
    {
        $risks = RiskEvaluation::with('risk','risk.iperc.place', 'risk.iperc.process', 'risk.iperc.task', 'risk.iperc.workPosition','risk.iperc.danger')->get();

        $data1 = $risks->map(function ($risk) {
            $iperc = $risk->risk->iperc->first(); // Tomar el primer IPERC asociado al riesgo
            return [
                'risk' => $risk->risk->description,
                'iperc_type' => $iperc ? $iperc->type : null,
                'iperc_routine' => $iperc ? $iperc->routine : null,
                'place' => $iperc ? $iperc->place->name : null,
                'process' => $iperc ? $iperc->process->name : null,
                'task' => $iperc ? $iperc->task->description : null,
                'work_position' => $iperc ? $iperc->workPosition->name : null,
                'danger' => $iperc ? $iperc->danger->description : null,
            ];
        });
        return response()->json($data1);
    }

    public function getRecommendation($id, $type)
    {
        $risk = Risk::find($id);
        $prompt = "Analizar las siguiente variable para hacer sugerencias de evaluación de riesgo bajo el esquema SSOMA:".
        "Riesgo asociado: " . $risk->description . ". Tipo de riesgo: ".($type=="inherent" ? "inherente": "residual").
        "Responde en formato json sólo con números del 0 al 3 y sin descripción introductoria, donde 0 es nulo, 1 es muy bajo y 3 es muy alto,
        siguiendo una estructura json: { index_a, index_b, index_c, index_d, severity }, donde index_a es el índice de personas expuestas (0 al 3),
        index_b es el indice de procedimientos existentes (0 al 3),
        index_c es el indice de capacitación (0 al 3), index_d es el índice de exposición al riesgo (0 al 3) y severity es el índice de severidad (1 al 3).
        json=";

        $response = $this->envia($prompt);

        $content = $response['choices'][0]['message']['content'];
        return response()->json($content);
    }

    public function envia($param)
    {
        try {
            $messages = [
                [
                    "role" => "user",
                    "content" => $param
                ]
            ];
            $apiKey = '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';

            $data = [
                "model" => "gpt-3.5-turbo",
                'messages' => $messages,
                'temperature' => 0.7,
                'max_tokens' => 700,
                'n' => 1,
                'stop' => ['\n']
            ];

            $ch = curl_init('https://api.openai.com/v1/chat/completions');
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                'Content-Type: application/json',
                'Authorization: Bearer ' . $apiKey
            ));

            $response = curl_exec($ch);
            $responseArr = json_decode($response, true);

            if (isset($responseArr['choices']) && is_array($responseArr['choices']) && count($responseArr['choices']) > 0) {
                $data = $responseArr;
            } else {
                $data = $responseArr;
            }

            return $data;

        } catch (\Throwable $th) {
            // Registrar el error
            Log::info("Ocurrió un error al consultar", [$th->getMessage()]);
        }
    }

}
