<?php

namespace App\Http\Controllers;

use App\Helpers\Helper;
use App\Models\Alert;
use App\Models\News;
use App\Models\NewsTagDetail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class NewsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('news');
    }

    public function index()
    {
        return view('content.news.index');
    }

    public function update(Request $request)
    {
        try {
            $news = News::findOrFail($request->id);
            $news->title = $request->title;
            $news->description = $request->description;
            if ($request->hasFile('thumbnail')) {
                $file = $request->file('thumbnail');
                $portada = Helper::formatFilename($file->getClientOriginalName());
                $path = 'news/' . $request->id . '/' . 'portada/';
                Storage::disk('s3')->put($path . $portada, file_get_contents($file), 'public');
                $news->thumbnail = $path . $portada;
            }
            $news->content = $request->content;
            $news->slug = Str::slug($request->title);
            if ($news->update()) {
                $tags = $request->tags; // array of tags
                $oldTags = NewsTagDetail::where('new_id', $request->id)->pluck('tag_id')->toArray();
                $this->removeDetailTagsIfNotInUpdate($oldTags, $tags, $request->id);
                foreach ($tags as $tag) {
                    $castTag = json_decode($tag);
                    // if tag detail doesnt exists
                    if (!$this->detailTagExists($castTag->id, $request->id)) {
                        $detail = new NewsTagDetail();
                        $detail->tag_id = $castTag->id;
                        $detail->new_id = $news->id;
                        $detail->save();
                    }
                }
                DB::commit();
                return response()->json(['success' => 'Noticia editada correctamente']);
            } else {
                DB::rollBack();
                return response()->json(['error' => 'Error al editar la noticia']);
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function detailTagExists($tagId, $newsId)
    {
        return NewsTagDetail::where('tag_id', $tagId)->where('new_id', $newsId)->exists();
    }

    public function removeDetailTagsIfNotInUpdate($oldTags, $tags, $newId)
    {
        $newTags = [];
        foreach ($tags as $tag) {
            $castTag = json_decode($tag);
            array_push($newTags, $castTag->id);
        }

        $removeValues = array_diff($oldTags, $newTags);
        foreach ($removeValues as $value) {
            NewsTagDetail::where('tag_id', $value)->where('new_id', $newId)->delete();
        }
    }


    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $query = DB::select("SHOW TABLE STATUS LIKE 'news'");
            $nextId = $query[0]->Auto_increment;

            $news = new News();
            $news->title = $request->title;
            $news->description = $request->description;
            if ($request->hasFile('thumbnail')) {
                $file = $request->file('thumbnail');
                $portada = Helper::formatFilename($file->getClientOriginalName());
                $path = 'news/' . $nextId . '/' . 'portada/';
                Storage::disk('s3')->put($path . $portada, file_get_contents($file), 'public');
                $news->thumbnail = $path . $portada;
            }
            $news->content = $request->content;
            $news->slug = Str::slug($request->title);
            $news->author_id = auth()->user()->id;
            $news->status = 1; //estado de publicacion 0:no publicar | 1:publicar
            if ($news->save()) {
                $tags = $request->tags; // array of tags
                foreach ($tags as $tag) {
                    $castTag = json_decode($tag);
                    $detail = new NewsTagDetail();
                    $detail->tag_id = $castTag->id;
                    $detail->new_id = $news->id;
                    $detail->save();
                }
                DB::commit();
                return response()->json(['success' => 'Noticia creada correctamente']);
            } else {
                DB::rollBack();
                return response()->json(['error' => 'Error al crear la noticia']);
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function updateStatus(Request $request)
    {
        try {
            $news = News::findOrFail($request->id);
            $news->status = $request->status;
            if ($news->update()) {
                return response()->json(['success' => 'Estado de la noticia actualizado correctamente']);
            } else {
                return response()->json(['error' => 'Error al actualizar el estado de la noticia']);
            }
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function list()
    {
        $news = News::with(['tags', 'author'])
            ->get();
        return response()->json($news);
    }

    public function changeStatus(Request $request)
    {
        $news = News::findOrFail($request->id);
        $news->status = $news->status == 0 ? 1 : ($news->status == 1 ? 0 : 0);
        $news->update();
        return $news->status;
    }
}
