<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Clas;
use App\Models\User;
use App\Models\Badge;
use App\Models\Video;
use App\Models\Course;
use App\Models\Module;
use App\Helpers\Helper;
use App\Models\Category;
use App\Models\BadgeDetail;
use App\Models\CourseLevel;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\Notifications;
use App\Models\PurchasedCourse;
use App\Models\CourseObservation;
use App\Models\UserConfiguration;
use App\Traits\CourseProcessTrait;
use Illuminate\Support\Facades\DB;
use App\Models\CourseConfiguration;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class CourseController extends Controller
{
    use CourseProcessTrait;

    public function __construct()
    {
        $this->middleware('can:courses.create')->only('create');
        $this->middleware('can:courses.index')->only('index', 'listCoursesProd');
        $this->middleware('can:courses.edit')->only('edit');
    }

    public function index()
    {
        $user = User::find(auth()->user()->id);
        $permission = $user->hasPermissionTo('courses.create');
        return view('content.courses.index', compact('user', 'permission'));
    }

    // public function publishedCourses()
    // {
    //     $user = User::find(auth()->user()->id);
    //     $permission = $user->hasPermissionTo('courses.create');
    //     return view('content.courses.index', compact('user', 'permission'));
    // }

    public function create()
    {
        $categories = Category::all();
        $levels = CourseLevel::all();
        return view('content.courses.create', compact('categories', 'levels'));
    }

    public function review($id)
    {
        $course = Course::where('courses.id', $id)->join('categories', 'courses.id_categories', '=', 'categories.id')->join('course_level', 'courses.course_level_id', '=', 'course_level.id')->select('courses.*', 'course_level.description as level', 'categories.name as category')->get()->first();
        $ready = $this->isReadyToPublish($course->id);
        return view('content.courses.verification.review', compact('course', 'ready'));
    }

    public function isReadyToPublish($id_course)
    {
        /**
         * Estados de la clase
         * 0 = clase no revisada
         * 1 clase desaprobada
         */
        // Get id of all modules
        $modules_ids = Course::find($id_course)->modules->pluck('id')->toArray();
        $lessons_status = [];
        foreach ($modules_ids as $module_id) {
            $status_of_lessons_from_module = Clas::where('id_modules', $module_id)->pluck('status')->toArray();
            array_push($lessons_status, $status_of_lessons_from_module);
        }
        $status_array = [];
        // Get status of all lessons in all modules
        foreach ($lessons_status as $status) {
            foreach ($status as $s) {
                array_push($status_array, $s);
            }
        }
        // si existe clase desaprobada o no ha sido revisada retorna false
        if (in_array(1, $status_array) || in_array(0, $status_array)) {
            return false;
        } else {
            return true;
        }
    }

    public function approved($course_id)
    {
        /**
         * estados del curso 
         * 0 -> creado
         * 1 -> enviado a revision
         * 2 -> aprobado
         * 3 -> curso con observaciones
         */
        try {
            DB::beginTransaction();
            $course = Course::findOrFail($course_id);
            $courseVal = $course;
            $course->status = 2;
            $course->update();

            $title = 'Curso aprobado';
            $body = "$course->title fue aprobado!";
            $this->notification($course->user_id, $title, $body);

            $productor_id = $course->user_id;
            $badge_level_one_id = 10;
            $userHasBadge1 = $this->validateIfUserHasBadge($badge_level_one_id, $productor_id);
            if ($userHasBadge1 == false) {
                $goal = Badge::where('id', $badge_level_one_id)->get()->first()->condition;
                $course_count = Course::where(['user_id' => $productor_id, 'status' => 2])->count();
                if ($course_count >= $goal) {
                    $badge = new BadgeDetail();
                    $badge->user_id = $productor_id;
                    $badge->badge_id = $badge_level_one_id;
                    $badge->save();
                    $title = 'Logro desbloqueado';
                    $body = 'Ha conseguido el logro por crear un curso';
                    $this->notification($course->user_id, $title, $body);
                }
            }
            $badge_level_two_id = 11;
            $userHasBadge2 = $this->validateIfUserHasBadge($badge_level_two_id, $productor_id);
            if ($userHasBadge2 == false) {
                $goal = Badge::where('id', $badge_level_two_id)->get()->first()->condition;
                $course_count = Course::where(['user_id' => $productor_id, 'status' => 2])->count();
                if ($course_count >= $goal) {
                    $badge = new BadgeDetail();
                    $badge->user_id = $productor_id;
                    $badge->badge_id = $badge_level_two_id;
                    $badge->save();
                    $title = 'Logro desbloqueado';
                    $body = "Ha conseguido el logro por crear $goal cursos";
                    $this->notification($course->user_id, $title, $body);
                }
            }
            $badge_level_three_id = 12;
            $userHasBadge3 = $this->validateIfUserHasBadge($badge_level_three_id, $productor_id);
            if ($userHasBadge3 == false) {
                $goal = Badge::where('id', $badge_level_three_id)->get()->first()->condition;
                $course_count = Course::where(['user_id' => $productor_id, 'status' => 2])->count();
                if ($course_count >= $goal) {
                    $badge = new BadgeDetail();
                    $badge->user_id = $productor_id;
                    $badge->badge_id = $badge_level_three_id;
                    $badge->save();
                    $title = 'Logro desbloqueado';
                    $body = "Ha conseguido el logro por crear $goal cursos";
                    $this->notification($course->user_id, $title, $body);
                }
            }
            DB::commit();
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function validateIfUserHasBadge($badge_id, $user_id)
    {
        $bool = BadgeDetail::where(['user_id' => $user_id, 'badge_id' => $badge_id])->exists();
        return $bool;
    }

    public function notification($id_user, $title, $body)
    {
        try {
            DB::beginTransaction();
            $notification = new Notifications();
            $notification->id_generator = $id_user;
            $notification->id_receiver =  $id_user;
            $notification->title = $title;
            $notification->body = $body;
            $notification->type = 3; # Compra de cursos
            $notification->save();
            DB::commit();

        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }
    // cambiamos los estados de las observaciones de un curso para que pueda visualizarlas el productor
    // cambiar el estado del curso a curso con observaciones
    public function sendObservations($course_id)
    {
        // las observaciones deben enviarse al terminar de revisar todo el curso
        // validar que todas las clases hayan sido revisadas de lo contrario se enviará un mensaje que diga que falta revisar clases
        $isReadyToSendObservation = $this->isReadyToSendObservation($course_id);
        if ($isReadyToSendObservation === true) {
            $observations = CourseObservation::where('id_course', $course_id)->get();
            $course = Course::where('id', $course_id)->get()->first();
            $course->status = 3;
            $course->update();
            $title = 'Curso con observaciones';
            $body = "$course->title tiene observaciones.";
            $this->notification($course->user_id, $title, $body);
            try {
                DB::beginTransaction();
                foreach ($observations as $observation) {
                    $observation->status = 1;
                    $observation->update();
                }

                $response['status'] = 'ok';
                DB::commit();
            } catch (\Throwable $th) {
                DB::rollBack();
                throw $th;
            }
        } else {
            // Hay clases por revisar aun
            $response['status'] = 'pending';
        }
        echo json_encode($response);
    }

    // Valida que todas las clases tengan un estado distinto a 0 que significa que no se reviso
    public function isReadyToSendObservation($course_id)
    {
        $modules_ids = Course::find($course_id)->modules->pluck('id')->toArray();
        foreach ($modules_ids as $module) {
            $status = Clas::where('id_modules', $module)->pluck('status')->toArray();
        }
        if (in_array(0, $status)) {
            return false;
        } else {
            return true;
        }
    }



    public function storeCourse(Request $request)
    {
        $id = DB::select("SHOW TABLE STATUS LIKE 'courses'");
        $next_id = $id[0]->Auto_increment;
        $user = Auth::user();
        try {
            DB::beginTransaction();
            $course = new Course();
            $course->user_id = $user->id;
            $course->id_categories = $request->id_categories;
            $course->title = $request->title;
            $course->slug = Str::slug($request->title);
            $course->description = $request->description;
            $course->price_base = $request->price_base;
            $course->price = $request->price;
            $course->course_level_id = $request->id_level;
            $course->course_about = $request->course_about;
            $course->will_learn = $request->will_learn;
            $course->prev_knowledge = $request->prev_knowledge;
            $course->course_for = $request->course_for;
            $course->currency = 'soles';
            $course->months = $request->months;
            $course->certificate = $request->certificate === "true" ? 1 : 0;
            $course->status = 0;

            if ($request->hasFile('file')) {
                $file = $request->file('file');
                $portada = Helper::formatFilename($file->getClientOriginalName());
                $path = 'courses/' . $user->id  . '/' . $next_id . '/' . 'portada/';
                Storage::disk('s3')->put($path . $portada, file_get_contents($file), 'public');
                $course->portada = $portada;
                $course->url_portada = $path . $portada;
            }

            if ($request->hasFile('file_video')) {
                $file = $request->file('file_video');
                $portada = Helper::formatFilename($file->getClientOriginalName());
                $path = 'courses/' . $user->id  . '/' . $next_id . '/' . 'portada/';
                Storage::disk('s3')->put($path . $portada, file_get_contents($file), 'public');
                $course->path_url = $path . $portada;
            }

            if ($course->save()) {
                $response['status'] = 'ok';
            } else {
                $response['status']  = 'error';
            }

            DB::commit();
            
            return response()->json([
                'data' => $response,
                'message' => 'Data recuperada con exito'
            ], 200);
            
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function courseList($id)
    {
        $courses = Course::where('user_id', $id)->join('course_level', 'courses.course_level_id', '=', 'course_level.id')->select('courses.*', 'course_level.description as level')->orderBy('id', 'DESC')->get();
        
        return response()->json([
            'data' => $courses,
            'message' => 'Data recuperada con exito',
        ], 200);
    }

    public function courseListVerification()
    {
        $courses = Course::where('status', 1)->join('users', 'courses.user_id', '=', 'users.id')->select('courses.*', 'users.name')->get();
        
        return response()->json([
            'data' => $courses,
            'message' => 'Data recuperada con exito',
        ], 200);
    }

    public function edit($id)
    {
        $course = Course::where('id', $id)->join('course_level', 'courses.course_level_id', '=', 'course_level.id')->select('courses.*', 'course_level.description as level')->get()->first();
        echo json_encode($course);
    }

    public function update($id, Request $request)
    {
        $user = User::find(auth()->user()->id);
        $course = Course::where('id', $request->id_course)->get()->first();

        try {
            DB::beginTransaction();
            $course = Course::where('id', $id)->get()->first();
            $course->user_id = $user->id;
            $course->id_categories = $request->id_categories;
            $course->title = $request->title;
            $course->slug = Str::slug($request->title);
            $course->description = $request->description;
            $course->price_base = $request->price_base;
            $course->price = $request->price;
            $course->course_level_id = $request->id_level;
            $course->course_about = $request->course_about;
            $course->will_learn = $request->will_learn;
            $course->prev_knowledge = $request->prev_knowledge;
            $course->course_for = $request->course_for;
            $course->months = $request->months;
            $course->certificate = $request->certificate === "true" ? 1 : 0;
            $course->currency = 'soles';

            if ($request->hasFile('file')) {
                $newfile = $request->file('file');
                $portada = Helper::formatFilename($newfile->getClientOriginalName());
                $path = 'courses/' . $user->id  . '/' . $id . '/' . 'portada/';
                Storage::disk('s3')->delete($course->url_portada);
                Storage::disk('s3')->put($path . $portada, file_get_contents($newfile), 'public');
                $course->portada = $portada;
                $course->url_portada = $path . $portada;
            }

            if ($request->hasFile('file_video')) {
                $newfile = $request->file('file_video');
                $portada = Helper::formatFilename($newfile->getClientOriginalName());
                $path = 'courses/' . $user->id  . '/' . $id . '/' . 'portada/';
                Storage::disk('s3')->delete($course->url_portada);
                Storage::disk('s3')->put($path . $portada, file_get_contents($newfile), 'public');
                $course->path_url = $path . $portada;
            }

            if ($course->update()) {
                $response['status'] = 'ok';
            } else {
                $response['status']  = 'error';
            }
            echo json_encode($response);
            DB::commit();
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function listCoursesProd()
    {
        $user = User::find(auth()->user()->id);
        $data['course'] = $user->MyCourses()->get();
        $data['latests'] = Course::where('user_id', $user->id)->latest()->first();
        if (count($data['course']) == 0) {
            return ['error' => 'user without courses'];
        } else {
            return $data;
        }
    }

    public function delete($id)
    {
        $user = Auth::user();
        $course = Course::where('id', $id)->first();

        $modules = Course::find($id)->modules;
        if (count($modules)) {
            foreach ($modules as $module) {
                CourseModuleController::delete($module->id);
            }
        }
        try {
            DB::beginTransaction();
            if ($course->delete()) {
                $response['courses'] = Course::where('user_id', $user->id)->get();
                $response['status'] = 'ok';
            } else {
                $response['status'] = 'error';
            }
            echo json_encode($response);
            DB::commit();
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function changeState($id)
    {
        try {
            DB::beginTransaction();
            $course = Course::findOrFail($id);
            $course->status = ($course->status === 0) ? 1 : 0;

            if ($course->update()) {
                $user = User::find(auth()->user()->id);
                $response['courses'] = $user->MyCourses()->get();
                $response['status'] = 'ok';
            } else {
                $response['status']  = 'error';
            }
            echo json_encode($response);
            DB::commit();
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function userCourses($user_id)
    {
        $courses = Course::where('user_id', $user_id)->join('course_level', 'courses.course_level_id', '=', 'course_level.id')->select('courses.*', 'course_level.description as level')->get();
        return $courses;
    }

    public function modulesList($course_id)
    {
        $modules = Course::find($course_id)->modules;
        
        return response()->json([
            'data' => $modules,
            'message' => 'Data recuperada con exito',
        ], 200);
    }

    public function verifyIsHasModules($course_id)
    {
        $modules = Module::where('id_courses', $course_id)->get();
        if (count($modules) > 0) {
            $first_class = Clas::where('id_modules', $modules[0]->id)->get();
            if (count($first_class) > 0) {
                return true;
            }
        }
        return false;
    }

    public function prueba(){
        $signatureConfig = UserConfiguration::where('user_id', 2)
                        ->where('configuration_id', 2)->exists();
                    $templateConfig = UserConfiguration::where('user_id', 5)
                        ->where('configuration_id', 1)->exists();
                    $config = $signatureConfig && $templateConfig ? true : false;
                    return $templateConfig;
    }
    
    public function sendRequest($id)
    {
        try {
            DB::beginTransaction();
            $course = Course::findOrFail($id);
            $hasModules = $this->verifyIsHasModules($id);

            if (!$hasModules) {
                $response = 'empty';
            } else {
                $response = $this->processCourse($course);
            }

            DB::commit();

            return response()->json([
                'data' => $response,
                'message' => 'Data recuperada con exito',
            ], 200);
            
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function categoriesList()
    {
        $categories = Category::all();
        return $categories;
    }

    public function levelsList()
    {
        $levels = CourseLevel::all();
        return $levels;
    }

    public function subscribers()
    {
        $id = auth()->user()->id;
        $courses = Course::where('user_id', $id)->select('id', 'id_categories', 'title', 'description', 'url_portada', 'portada')->get();
        $path = 'https://promolider-storage-user.s3-accelerate.amazonaws.com/';

        $shareComponent = \Share::page('http://promolider.xyz/login')->facebook()
            ->twitter()
            ->linkedin()
            ->telegram()
            ->whatsapp()
            ->reddit();


        return view('content.courses.subscriber.index', compact('courses', 'path', 'shareComponent'));
    }

    public function subscribersList($course_id)
    {
        $users = [];
        $subscriber = PurchasedCourse::where('course_id', $course_id)->get();
        foreach ($subscriber as $index => $value) {
            $user = User::where('id', $value->user_id)->get()->first();
            $user->completed_course = $value->completed_course;
            $user->certificate_delivered = $value->certificate_delivered;
            $user->purchased_course_id = $value->id;
            array_push($users, $user);
        }

        return response()->json([
            'data' => $users,
            'message' => 'Data recuperada con exito',
        ], 200);
    }

    public function verification()
    {
        $user = User::find(auth()->user()->id);
        return view('content.courses.verification.index', compact('user'));
    }

    public function configureCertificate($course_id)
    {
        $course = Course::select('id', 'title')->where('id', $course_id)->get()->first();
        $courseConfig= CourseConfiguration::where('id', $course_id)->get()->first();
        return view('content.courses.certificate.index', compact('course'));
    }

    public function getConfigureCertificate($course_id)
    {
        $courseConfig= CourseConfiguration::where('course_id', $course_id)->get()->first();
        return response()->json([
            'status' => true,
            'data' => $courseConfig,
            'message' => 'Data recuperada con exito'
        ], 200);
    }

    public function getOrders($id){
        $data = collect();
        $modules = Course::join('modules', 'courses.id', '=', 'modules.id_courses')
            ->where('courses.id', $id)
            ->orderBy('modules.order', 'asc')
            ->select('modules.id', 'modules.name')
            ->get();
        foreach ($modules as $module) {
            $module->type = "module";
            $data->push($module);
            $classes = Clas::where('id_modules', $module->id)
                ->select('id', 'name', 'id_modules')
                ->orderBy('order', 'asc')
                ->get();
            if(!$classes->isEmpty()){
                foreach ($classes as $class) {
                    $class->type = "class";
                    $videos = Video::where('class_id', $class->id)->get();
                    $class->videos = $videos;
                    $data->push($class);
                }
            }
        }
        return $data;
    }

    public function changeOrder(Request $request){
        $items = json_decode($request->order, true);
        $i = 1;
        $item_id = 0;
        foreach ($items as $item) {
            $type = $item['type'];
            if($type == "module"){
                $item_id = $item['id'];
            }else{
                $item_id = $item['id'];
                $class = Clas::findOrFail($item_id);
                $class->order = $i;
                $class->update();
            }
            $i++;
        }
        return $items;
    }

    public function changeOrderModule(Request $request)
    {
        $items = json_decode($request->order, true);
        $i = 1;
        $item_id = 0;
        foreach ($items as $item) {
            $item_id = $item['id'];
            $module = Module::findOrFail($item_id);
            $module->order = $i;
            $module->update();
           
            $i++;
        }
        return $items;
    }
}
